<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Http\Client\RequestException;

class WhatsAppService
{
    protected ?string $phoneNumberId;
    protected ?string $accessToken;
    protected string $apiVersion;
    protected string $baseUrl;

    public function __construct()
    {
        // $this->phoneNumberId = config('whatsapp.phone_number_id');
        // $this->accessToken   = config('whatsapp.access_token');
        // $this->apiVersion    = config('whatsapp.api_version', 'v20.0');

        $this->phoneNumberId = 880579065146537;
        $this->accessToken   = "EAFnhJeQ8Re4BQMjvUbmWjpZBZBZCfrdSdgNiWjQDTLfRASGaMchlZAXDzUdvZCCWApVgZBKvfTKpE5w3RCAUZBi45hmhsfTromfRkq4w1SzSZBxvjJNIdLG6xWE1NdtXZCzMXFCqUBcNDcxdRmtgBtZB2ftZB5fatO98Vs6a8A6JdPsQF62M20nTO2dYR7D7BT29gqfXAZDZD";
        $this->apiVersion    = "v20.0";

        // Only set baseUrl if phoneNumberId is not null
        if ($this->phoneNumberId) {
            $this->baseUrl = "https://graph.facebook.com/{$this->apiVersion}/{$this->phoneNumberId}";
        } else {
            $this->baseUrl = '';
        }
    }

    /**
     * Check if WhatsApp service is properly configured
     */
    public function isConfigured(): bool
    {
        return !empty($this->phoneNumberId) && !empty($this->accessToken);
    }

    /**
     * Validate configuration before making API calls
     */
    private function validateConfiguration(): void
    {
        if (!$this->isConfigured()) {
            throw new \Exception('WhatsApp service is not properly configured. Please check phone_number_id and access_token in config.');
        }
    }

    /**
     * Generic method to send a template message
     */
    // public function sendTemplateMessage(string $to, string $templateName, string $languageCode = 'en_US', array $components = [])
    // {
    //     $this->validateConfiguration();
    //     $payload = [
    //         'messaging_product' => 'whatsapp',
    //         'to'   => $to,
    //         'type' => 'template',
    //         'template' => [
    //             'name'     => $templateName,
    //             'language' => [
    //                 'code' => $languageCode,
    //             ],
    //         ],
    //     ];

    //     if (!empty($components)) {
    //         $payload['template']['components'] = $components;
    //     }

    //     $response = Http::withToken($this->accessToken)
    //         ->post("{$this->baseUrl}/messages", $payload);

    //     if ($response->failed()) {
    //         // Optional: throw exception or log it
    //         throw new RequestException($response);
    //     }

    //     return $response->json();
    // }

    /**
     * Example: media upload (similar to MEDIA_URL in Java)
     */
    public function uploadMedia(string $filePath, string $mimeType = 'image/jpeg')
    {
        $this->validateConfiguration();
        
        $response = Http::asMultipart()
            ->withToken($this->accessToken)
            ->attach('file', file_get_contents($filePath), basename($filePath))
            ->post("{$this->baseUrl}/media", [
                'messaging_product' => 'whatsapp',
            ]);

        if ($response->failed()) {
            throw new RequestException($response);
        }

        return $response->json();        // returns ['id' => 'MEDIA_ID']
    }

    /**
     * Convenience method: your plan_expire_reminder template
     * (ported from WhatsAppPayloads.planExpiredReminder)
     */
    public function sendPlanExpireReminder(array $planExpireInfo)
    {
        // $planExpireInfo = [
        //   'mobile_number' => '91XXXXXXXXXX',
        //   'name'          => 'Customer Name',
        //   'plan_name'     => 'Gold Plan',
        //   'expire_in_days'=> '3 days',
        // ];

        $components = [
            // Header with image: https://sarakuapp.com/expire-plan.png
            [
                'type' => 'header',
                'parameters' => [
                    [
                        'type'  => 'image',
                        'image' => [
                            'link' => 'https://sarakuapp.com/expire-plan.png',
                        ],
                    ],
                ],
            ],

            // Body with 3 text params: name, planName, expireInDays
            [
                'type' => 'body',
                'parameters' => [
                    [
                        'type' => 'text',
                        'text' => $planExpireInfo['name'],
                    ],
                    [
                        'type' => 'text',
                        'text' => $planExpireInfo['plan_name'],
                    ],
                    [
                        'type' => 'text',
                        'text' => $planExpireInfo['expire_in_days'],
                    ],
                ],
            ],
        ];

        return $this->sendTemplateMessage(
            $planExpireInfo['mobile_number'],
            'plan_expire_reminder',
           
            $components, 'en_US',
        );
    }

public function sendTextMessage(string $to, string $message)
{
    $this->validateConfiguration();
    
    // Format phone number - ensure it starts with country code
    $formattedNumber = $this->formatPhoneNumber($to);
    
    $payload = [
        'messaging_product' => 'whatsapp',
        'to'   => $formattedNumber,
        'type' => 'text',
        'text' => [
            'body' => $message
        ],
    ];

    $response = Http::withToken($this->accessToken)
        ->post("{$this->baseUrl}/messages", $payload);

    if ($response->failed()) {
        throw new RequestException($response);
    }

    return $response->json();
}

public function sendHelloTemplate(string $to)
{
    $payload = [
        'messaging_product' => 'whatsapp',
        'to'   => $to,
        'type' => 'template',
        'template' => [
            'name' => 'hello_world', // or any APPROVED template name
            'language' => [
                'code' => 'en_US',
            ],
        ],
    ];

    $response = Http::withToken($this->accessToken)
        ->post("{$this->baseUrl}/messages", $payload);

    return $response->json();
}


/**
 * Format phone number for WhatsApp API
 * Ensures number starts with country code (91 for India)
 */
private function formatPhoneNumber(string $phoneNumber): string
{
    // Remove any spaces, dashes, or other characters
    $cleaned = preg_replace('/[^\d]/', '', $phoneNumber);
    
    // If number starts with 0, remove it
    if (substr($cleaned, 0, 1) === '0') {
        $cleaned = substr($cleaned, 1);
    }
    
    // If number doesn't start with country code, add it (assuming India +91)
    if (strlen($cleaned) === 10) {
        $cleaned = '91' . $cleaned;
    }
    
    return $cleaned;
}

public function sendTemplateMessage(
    string $to,
    string $templateName,
    array $bodyParams = [],
    string $languageCode = 'en_US'
): array {
    $this->validateConfiguration();

    $formattedNumber = $this->formatPhoneNumber($to);

    // Build body components from params
    $components = [];
    if (!empty($bodyParams)) {
        $bodyParameters = [];
        foreach ($bodyParams as $value) {
            $bodyParameters[] = [
                'type' => 'text',
                'text' => (string) $value,
            ];
        }

        $components[] = [
            'type' => 'body',
            'parameters' => $bodyParameters,
        ];
    }

    $payload = [
        'messaging_product' => 'whatsapp',
        'to'        => $formattedNumber,
        'type'      => 'template',
        'template'  => [
            'name'     => $templateName,   // e.g. "ps25_bulk_message"
            'language' => ['code' => $languageCode],
        ],
    ];

    if (!empty($components)) {
        $payload['template']['components'] = $components;
    }

    $response = Http::withToken($this->accessToken)
        ->post("{$this->baseUrl}/messages", $payload);

    if ($response->failed()) {
        throw new RequestException($response);
    }

    return $response->json();
}

/**
 * Bulk send template – one API call per recipient
 * $recipients = [
 *   ['mobile_number' => '91xxxx', 'body_params' => ['Abinaya', 'PS25']],
 *   ...
 * ];
 */
public function sendBulkTemplateMessages(
    array $recipients,
    string $templateName,
    string $languageCode = 'en_US'
): array {
    $results = [];

    foreach ($recipients as $row) {
        $mobile     = $row['mobile_number'] ?? null;
        $bodyParams = $row['body_params'] ?? [];

        if (!$mobile) {
            $results[] = [
                'mobile' => null,
                'status' => 'failed',
                'reason' => 'mobile_number missing',
            ];
            continue;
        }

        try {
            $resp = $this->sendTemplateMessage($mobile, $templateName, $bodyParams, $languageCode);

            $results[] = [
                'mobile'   => $mobile,
                'status'   => 'success',
                'response' => $resp,
            ];
        } catch (\Throwable $e) {
            $results[] = [
                'mobile' => $mobile,
                'status' => 'failed',
                'reason' => $e->getMessage(),
            ];
        }
    }

    return $results;
}


}
