<?php

namespace App\Repositories;

use App\Models\Voter;
use App\Models\RationCard;

class VoterRepository
{
    protected $voter;

    public function __construct(Voter $voter)
    {
        $this->voter = $voter;
    }

    /**
     * Create a new voter
     * 
     * @param array $data
     * @return Voter|null
     */
    public function createVoter(array $data)
    {
        // Check if voter ID already exists
        if (isset($data['voter_id_number'])) {
            $existingVoter = $this->voter::where('voter_id_number', $data['voter_id_number'])
                                        ->where('is_deleted', false)
                                        ->first();
            
            if ($existingVoter) {
                return null; // Voter ID already exists
            }
        }

        // Set default values
        $data['is_deleted'] = false;

        return $this->voter::create($data);
    }

    /**
     * Get voter by ID
     * 
     * @param int $voterId
     * @return Voter|null
     */
    public function getVoterById($voterId)
    {
        return $this->voter::where('id', $voterId)
                          ->where('is_deleted', false)
                          ->first();
    }

    /**
     * Get voter by voter ID number
     * 
     * @param string $voterIdNumber
     * @return Voter|null
     */
    public function getByVoterIdNumber($voterIdNumber)
    {
        return $this->voter::where('voter_id_number', $voterIdNumber)
                          ->where('is_deleted', false)
                          ->first();
    }

    /**
     * Check if ration card exists
     * 
     * @param string $rationCardNumber
     * @return RationCard|null
     */
    public function getRationCardByNumber($rationCardNumber)
    {
        return RationCard::where('ration_card_number', $rationCardNumber)
                        ->where('is_deleted', false)
                        ->first();
    }

    /**
     * Get all voters
     * 
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getAllVoters()
    {
        return $this->voter::where('is_deleted', false)->get();
    }

    /**
     * Get voter count
     * 
     * @return int
     */
    public function getVoterCount()
    {
        return $this->voter::where('is_deleted', false)->count();
    }

    /**
     * Update voter
     * 
     * @param int $voterId
     * @param array $data
     * @return Voter|null
     */
    public function updateVoter($voterId, array $data)
    {
        $voter = $this->getVoterById($voterId);
        
        if (!$voter) {
            return null;
        }

        // Check if voter ID is being updated and if it conflicts with existing
        if (isset($data['voter_id_number']) && $data['voter_id_number'] !== $voter->voter_id_number) {
            $existingVoter = $this->getByVoterIdNumber($data['voter_id_number']);
            if ($existingVoter) {
                return null; // Voter ID already exists
            }
        }

        $voter->update($data);
        return $voter;
    }

    /**
     * Soft delete voter
     * 
     * @param int $voterId
     * @return bool
     */
    public function deleteVoter($voterId)
    {
        $voter = $this->getVoterById($voterId);
        
        if ($voter) {
            $voter->is_deleted = true;
            return $voter->save();
        }

        return false;
    }

    /**
     * Map existing voter to ration card
     * 
     * @param string $voterIdNumber
     * @param int $rationCardId
     * @return array
     */
    public function mapVoterToRationCard($voterIdNumber, $rationCardId)
    {
        // Find voter by voter_id_number
        $voter = $this->getByVoterIdNumber($voterIdNumber);
        
        if (!$voter) {
            return [
                'success' => false, 
                'message' => "Voter not found with voter ID number: {$voterIdNumber}. Please verify the voter ID and try again.",
                'error_type' => 'voter_not_found'
            ];
        }

        // Check if voter is already mapped to another ration card
        if ($voter->ration_card_id && $voter->ration_card_id != $rationCardId) {
            $existingRationCard = $this->getRationCardById($voter->ration_card_id);
            $existingCardNumber = $existingRationCard ? $existingRationCard->ration_card_number : 'Unknown';
            return [
                'success' => false, 
                'message' => "Voter {$voterIdNumber} is already mapped to ration card: {$existingCardNumber}",
                'error_type' => 'already_mapped_different'
            ];
        }

        // Check if voter is already mapped to the same ration card
        if ($voter->ration_card_id == $rationCardId) {
            return [
                'success' => false, 
                'message' => 'Voter is already mapped to this ration card',
                'error_type' => 'already_mapped_same'
            ];
        }

        // Verify ration card exists
        $rationCard = $this->getRationCardByNumber(null); // We need to get by ID instead
        
        // Map voter to ration card
        $voter->ration_card_id = $rationCardId;
        $saved = $voter->save();

        if ($saved) {
            return [
                'success' => true, 
                'message' => "Voter {$voterIdNumber} successfully mapped to ration card", 
                'voter' => $voter
            ];
        } else {
            return [
                'success' => false, 
                'message' => 'Failed to map voter to ration card. Please try again.',
                'error_type' => 'save_failed'
            ];
        }
    }

    /**
     * Get ration card by ID
     * 
     * @param int $rationCardId
     * @return RationCard|null
     */
    public function getRationCardById($rationCardId)
    {
        return RationCard::where('id', $rationCardId)
                        ->where('is_deleted', false)
                        ->first();
    }
}