# Grievance Management - Quick Reference

## 🎯 Key Features Implemented

✅ **Tab-based Status Filter** - Get grievances sorted by nearest target_resolve_by date
✅ **Single Create/Update Form** - Use same screen for both operations
✅ **Optional Voter/Ration Card Links** - Both can be null
✅ **Smart Sorting** - Nearest deadlines first, then by reported date
✅ **Auto-behaviors** - Auto-set closed_at, reported_at, default status

## 📊 Status Values

```
OPEN          → Newly reported (default)
IN_PROGRESS   → Being worked on
CLOSED        → Successfully resolved
UNRESOLVED    → Could not be resolved
HOLD          → Temporarily on hold
```

## 🔗 API Endpoints Quick List

```
POST   /api/grievance/create
PUT    /api/grievance/update/{id}
GET    /api/grievance/get/{id}
GET    /api/grievance/getAll?status=OPEN
GET    /api/grievance/getByStatus/{status}        ⭐ For Tab Bar
GET    /api/grievance/getByVoterId/{voterId}
GET    /api/grievance/getByRationCard/{number}
DELETE /api/grievance/delete/{id}
GET    /api/grievance/statuses
```

## 📝 Create Request

```json
{
  "voter_id": "XDQ0110387",           // Optional
  "ration_card_number": "12345678",   // Optional
  "title": "Issue title",             // Required (max 160)
  "description": "Details...",        // Optional
  "status": "OPEN",                   // Optional (default: OPEN)
  "meeting_date": "2025-11-10",       // Optional
  "target_resolve_by": "2025-11-15",  // Optional
  "reported_at": "2025-11-04 10:30:00" // Optional (default: now)
}
```

## 🔄 Update Request

```json
{
  "status": "IN_PROGRESS",           // Changes status
  "meeting_date": "2025-11-12",      // Updates meeting date
  "target_resolve_by": "2025-11-20"  // Updates target date
}
```

When status → CLOSED, `closed_at` is auto-set to current timestamp.

## 📱 UI Tab Bar Implementation

```javascript
const tabs = [
  { label: 'Open', status: 'OPEN', 
    endpoint: '/api/grievance/getByStatus/OPEN' },
  { label: 'In Progress', status: 'IN_PROGRESS', 
    endpoint: '/api/grievance/getByStatus/IN_PROGRESS' },
  { label: 'Closed', status: 'CLOSED', 
    endpoint: '/api/grievance/getByStatus/CLOSED' },
  { label: 'Unresolved', status: 'UNRESOLVED', 
    endpoint: '/api/grievance/getByStatus/UNRESOLVED' },
  { label: 'On Hold', status: 'HOLD', 
    endpoint: '/api/grievance/getByStatus/HOLD' }
];

// Each tab calls its endpoint to get sorted grievances
```

## 🎨 Response Format

```json
{
  "status": 200,
  "message": "Grievances retrieved successfully",
  "data": {
    "grievances": [
      {
        "id": 1,
        "voter_id": "XDQ0110387",
        "ration_card_number": "12345678",
        "title": "Water supply issue",
        "description": "No water since last week",
        "status": "OPEN",
        "meeting_date": "2025-11-10",
        "reported_at": "2025-11-04 10:30:00",
        "target_resolve_by": "2025-11-15",
        "closed_at": null,
        "created_at": "2025-11-04T10:30:00.000000Z",
        "updated_at": "2025-11-04T10:30:00.000000Z",
        "voter": { /* voter details */ },
        "ration_card": { /* ration card details */ }
      }
    ],
    "count": 15,
    "status": "OPEN"
  }
}
```

## ⚡ Sorting Logic (getByStatus)

1. Grievances **with** `target_resolve_by` → sorted nearest first ⭐
2. Grievances **without** `target_resolve_by` → appear at end
3. Within each group → sorted by `reported_at` DESC (newest first)

## 🧪 Test Examples

```bash
# Create
curl -X POST http://localhost:8000/api/grievance/create \
  -H "Content-Type: application/json" \
  -d '{"title":"Test","status":"OPEN"}'

# Get by Status (for tab)
curl http://localhost:8000/api/grievance/getByStatus/OPEN

# Update
curl -X PUT http://localhost:8000/api/grievance/update/1 \
  -H "Content-Type: application/json" \
  -d '{"status":"IN_PROGRESS"}'

# Get by Voter
curl http://localhost:8000/api/grievance/getByVoterId/XDQ0110387

# Get all statuses
curl http://localhost:8000/api/grievance/statuses
```

## 📁 Files Created

```
database/migrations/
  └── 2025_11_04_000001_create_grievances_table.php

app/
  ├── Models/Grievance.php
  ├── Repositories/GrievanceRepository.php
  ├── Services/GrievanceService.php
  └── Http/Controllers/GrievanceController.php

routes/api.php (updated)

Documentation:
  ├── GRIEVANCE_API_DOCUMENTATION.md
  └── GRIEVANCE_IMPLEMENTATION_SUMMARY.md
```

## ✅ Validation Rules

| Field | Rule |
|-------|------|
| `title` | Required, max 160 chars |
| `voter_id` | Optional, string, max 255 |
| `ration_card_number` | Optional, string, max 255 |
| `description` | Optional, text |
| `status` | Optional, must be valid enum |
| `meeting_date` | Optional, valid date |
| `target_resolve_by` | Optional, valid date |
| `reported_at` | Optional, datetime (auto: now) |
| `closed_at` | Optional, datetime (auto: on close) |

---

**✨ Ready to Use!**
All endpoints tested and working. Database migrated successfully.
