# Gift Distribution API Documentation

## Feature Overview
The Gift Distribution Tracking system allows you to track which ration cards have received specific gifts from a trust. This prevents duplicate distributions and provides validation for invalid ration card numbers.

## API Endpoint
**POST** `/api/gifts/distribute`

## Request Format
```json
{
    "gift_number": "23",
    "ration_card_number": ["S323423", "SDFSDF", "RC001", "RC002"]
}
```

## Request Fields
- `gift_number` (required): The ID of the gift from the gifts table
- `ration_card_number` (required): Array of ration card numbers to distribute the gift to

## Response Format

### Successful Response
```json
{
    "status": 200,
    "message": "Gift distribution completed successfully",
    "data": {
        "successfully_distributed": 2,
        "total_requested": 4,
        "gift_name": "Rice Bag",
        "gift_id": "23",
        "validation_errors": {
            "invalidRationCards": ["SDFSDF"],
            "alreadyDistributed": ["RC001"]
        }
    }
}
```

### Error Response
```json
{
    "status": 404,
    "message": "Gift not found with ID: 23",
    "data": null
}
```

## Validation Rules

1. **Invalid Ration Cards**: Ration card numbers that don't exist in the `ration_cards` table or are marked as deleted
2. **Already Distributed**: Ration card numbers that have already received the specified gift
3. **Valid Records**: Only ration cards that pass both validations will be saved to `gifted_ration_cards` table

## Database Schema

### `gifted_ration_cards` Table
| Column | Type | Description |
|--------|------|-------------|
| id | bigint | Primary key |
| gift_id | bigint | Foreign key to gifts table |
| ration_card_number | varchar | Ration card number |
| is_deleted | boolean | Soft delete flag (default: false) |
| created_at | timestamp | Record creation time |
| updated_at | timestamp | Record update time |

## Usage Examples

### Example 1: All Valid Ration Cards
```bash
curl -X POST http://localhost:8000/api/gifts/distribute \
  -H "Content-Type: application/json" \
  -d '{
    "gift_number": "1",
    "ration_card_number": ["RC001", "RC002", "RC003"]
  }'
```

### Example 2: Mixed Valid and Invalid Cards
```bash
curl -X POST http://localhost:8000/api/gifts/distribute \
  -H "Content-Type: application/json" \
  -d '{
    "gift_number": "1", 
    "ration_card_number": ["RC001", "INVALID_CARD", "RC002"]
  }'
```

## Features Implemented

✅ **Database Migration**: Creates `gifted_ration_cards` table with proper indexes
✅ **Eloquent Model**: `GiftedRationCard` with relationships and helper methods
✅ **API Route**: POST `/api/gifts/distribute` 
✅ **Validation Logic**: Checks for invalid ration cards and duplicate distributions
✅ **Service Layer**: Business logic separated in `GiftService`
✅ **Controller**: Error handling and response formatting
✅ **Model Relationships**: Added relationships between Gift, RationCard, and GiftedRationCard

## Error Handling

The API handles various error scenarios:
- Invalid gift ID (404)
- Missing or invalid request data (400)
- Database connection issues (500)
- Validation errors are returned without stopping the process

## Performance Considerations

- Database indexes on frequently queried columns
- Unique constraint prevents duplicate entries
- Efficient bulk validation using array operations
- Proper foreign key relationships for data integrity